<?php

namespace App\Http\Controllers;

use Auth;
use App\Link;
use App\User;
use App\Category;
use App\Page;
use App\AdminSet;
use Illuminate\Http\Request;

class AdminSetController extends Controller
{
    // Admin visibility and access middleware
    public function __construct()
    {
        $this->middleware(['admin', 'auth']);
    }

    // Admin panel views
    public function admin_panel()
    {
        $links = Link::latest()->take(5)->get();
        $users = User::latest()->take(5)->get();
        $categories = Category::latest()->take(5)->get();

        return view('admin.admin_panel', compact('links', 'users', 'categories'));
    }

    public function pages()
    {
        $pages = Page::latest()->paginate(10);
        return view('admin.pages', compact('pages'));
    }

    // Show members list
    public function memlist()
    {
        $users = User::whereNull('type')->paginate(10);
        return view('admin.users', compact('users'));
    }

    // Show admins list
    public function adminlist()
    {
        $users = User::where('type', 'Admin')->paginate(10);
        return view('admin.admins', compact('users'));
    }

    // Member staff update
    public function memup(Request $request, User $user)
    {
        $user->type = request('type');
        $user->update();

        //redirect
        return redirect()->back()->withMessage('User role has updated successfully!');
    }

    // Add new site via admin panel area
    public function addnew(Request $request)
    {
        $validated = $this->validate($request, [
            'title' => 'required',
            'url' => 'max:50',
            'urlslug' => 'sometimes',
            'category_id' => 'required',
            'icon' => 'sometimes',
            'phone' => 'sometimes',
            'email' => 'sometimes',
            'address' => 'sometimes',
            'description' => 'required'
        ]);

        if ($request->hasFile('icon')) {
            $filename = time() . '.' . $request->icon->getClientOriginalExtension();
            $request->icon->storeAs('linkdir', $filename, 'public');

            $validated['icon'] = $filename;
        }

            // Remove http://
            $url = request('url');
            $url = preg_replace( "#^[^:/.]*[:/]+#i", "", $url );
            // $url = preg_replace("(^https?://)", "", $url);
            $validated['urlslug'] = $url;

        $validated['user_id'] = Auth::id();

        $link = Link::create($validated);

        return redirect()->route('adsites', compact('link'))->withMessage('A new record has added succesfully!');
    }

    // Show sites list
    public function sitelist()
    {
        $links = Link::latest()->paginate(10);
        $categories = Category::all();
        return view('admin.sites', compact('links', 'categories'));
    }

    // Update site
    public function update_site(Request $request, Link $link)
    {
        $this->validate($request, [
            'title' => 'required',
            'url' => 'max:50',
            'urlslug' => 'sometimes',
            'category_id' => 'required',
            'icon' => 'sometimes',
            'phone' => 'sometimes',
            'email' => 'sometimes',
            'address' => 'sometimes',
            'description' => 'required'
        ]);

        if ($request->hasFile('icon')) {
            $filename = time() . '.' . $request->icon->getClientOriginalExtension();
            $request->icon->storeAs('linkdir', $filename, 'public');

            $validated['icon'] = $filename;
        }

            // Remove http://
            $url = request('url');
            $url = preg_replace( "#^[^:/.]*[:/]+#i", "", $url );
            // $url = preg_replace("(^https?://)", "", $url);
            $validated['urlslug'] = $url;

        //update
        $link->update($request->all());

        //redirect
        return redirect()->route('adsites', $link->slug)->withMessage('Site description has updated successfully!');
    }

    // Delete site
    public function delete_site(Link $link)
    {
        $link->delete();
        return redirect()->back()->withMessage('The record deleted successfully!');
    }

    // Show category list
    public function catelist()
    {
        $categories = Category::paginate(10);
        return view('admin.create_category', compact('categories'));
    }

    // Show site information and social feilds with form
    public function settings()
    {
        $settings = AdminSet::first();
        return view('admin.settings', compact('settings'));
    }

    // Updating site info
    public function update_settings(Request $request, AdminSet $adminset )
    {
        $validated = $this->validate($request, [
            'limitpost' => 'required|numeric'
        ]);

        $adminset->title = request('title');
        $adminset->limitpost = request('limitpost');
        $adminset->description = request('description');        
        $adminset->about = request('about');
        $adminset->sitemsg = request('sitemsg');
        $adminset->rtitle = request('rtitle');
        $adminset->rmsg = request('rmsg');
        $adminset->update();

        //redirect
        return redirect()->route('adsettings', $adminset->id)->withMessage('Site info updated successfully!');
    }

    // Updating social page links
    public function update_social(Request $request, AdminSet $adminset )
    {
        $adminset->facebook = request('facebook');
        $adminset->twitter = request('twitter');
        $adminset->linkedin = request('linkedin');
        $adminset->blog = request('blog');
        $adminset->update();

        //redirect
        return redirect()->route('adsettings', $adminset->id)->withMessage('Social links updated successfully!');
    }

    // Show advertisement form
    public function ads()
    {
        $ads = AdminSet::first();
        return view('admin.ads', compact('ads'));
    }

    // Updating advertisements
    public function update_ads(Request $request, AdminSet $adminset )
    {
        $ads = AdminSet::first();
        $ads->leaderads = request('leaderads');
        $ads->sideads = request('sideads');
        
        $ads->update($request->all());

        //redirect
        return redirect()->route('ads', $ads->id)->withMessage('Advertisement updated successfully!');
    }

    // Support page view
    public function support()
    {
        return view('admin.supports');
    }
}
