<?php

namespace App\Http\Controllers;

use App\Link;
use App\Category;
use App\Comment;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LinkController extends Controller
{
    function __construct()
    {
         $this->middleware('auth', ['only' => ['create', 'store', 'edit', 'update', 'destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $categories = Category::all();
        $links = Link::latest()->paginate(10);

        return view('links.category', compact('categories', 'links'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $categories = Category::all();
        $links = Link::all();
        return view('links.create', compact('categories', 'links'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'title' => 'required|min:5|max:150',
            // Store the full URL; urlslug is derived.
            'url' => 'required|url|max:255',
            'urlslug' => 'sometimes',
            'category_id' => 'required',
            'icon' => 'sometimes|image|mimes:png,jpg,jpeg,gif,webp|max:2048',
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string|max:255',
            'description' => 'required|min:20'
        ]);

        // Icon add new
        if ($request->hasFile('icon')) {
            $filename = time() . '.' . $request->icon->getClientOriginalExtension();
            $request->icon->storeAs('icon', $filename, 'public');

            $validated['icon'] = $filename;
        }

        // Derive a normalized urlslug (host + optional path) for display.
        // Keep `url` intact (with scheme) so it's a valid clickable URL.
        $rawUrl = (string) $validated['url'];
        $parts = parse_url($rawUrl);
        $host = $parts['host'] ?? '';
        $path = $parts['path'] ?? '';
        $validated['urlslug'] = ltrim($host . $path, '/');

        $validated['user_id'] = Auth::id();

        $link = Link::create($validated);

        return redirect()->route('link.show', ['link' => $link->slug])->withMessage('Post added successfully!');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Link  $link
     * @return \Illuminate\Http\Response
     */
    public function show(Link $link)
    {
        // $link->load('category');
        // Comments and ratings with paginate
        $comments = $link->comments()->latest()->paginate(10);

        $link->increment('views');
        $c1 = Comment::where(['link_id'=>$link->id, 'stars'=>1])->count();
        $c2 = Comment::where(['link_id'=>$link->id, 'stars'=>2])->count();
        $c3 = Comment::where(['link_id'=>$link->id, 'stars'=>3])->count();
        $c4 = Comment::where(['link_id'=>$link->id, 'stars'=>4])->count();
        $c5 = Comment::where(['link_id'=>$link->id, 'stars'=>5])->count();
        
        $rated = Comment::where(['user_id'=>Auth::id(), 'link_id'=>$link->id])->first();

        // Calculate the stars and scores
        $sumscore = Comment::where('link_id', $link->id)->sum('score');
        $ratecount = $link->comments->count();

        if (is_float($link->star)) {
            $displaystars = round($link->star);
            if ($displaystars < $link->star) {
                $displaystars++;
            } else {
                $displaystars;
            }
            
        } else {
            $displaystars = $link->star;
        }
        
        
        // Show the progressbar
        $totalstars = Comment::where(['link_id'=>$link->id])->select('stars')->count();

        $star1 = Comment::where(['link_id'=>$link->id, 'stars'=>1])->select('stars')->count();
        $star2 = Comment::where(['link_id'=>$link->id, 'stars'=>2])->select('stars')->count();
        $star3 = Comment::where(['link_id'=>$link->id, 'stars'=>3])->select('stars')->count();
        $star4 = Comment::where(['link_id'=>$link->id, 'stars'=>4])->select('stars')->count();
        $star5 = Comment::where(['link_id'=>$link->id, 'stars'=>5])->select('stars')->count();

        if ($totalstars == 0) {
            $s1 = 0;
            $s2 = 0;
            $s3 = 0;
            $s4 = 0;
            $s5 = 0;
        } else {
            $s1 = $star1 / $totalstars * 100;
            $s2 = $star2 / $totalstars * 100;
            $s3 = $star3 / $totalstars * 100;
            $s4 = $star4 / $totalstars * 100;
            $s5 = $star5 / $totalstars * 100;
        }
        
        
        return view('links.single', compact('link', 'comments', 'rated', 'ratecount', 'displaystars', 's1', 's2', 's3', 's4', 's5', 'c1', 'c2', 'c3', 'c4', 'c5'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Link  $link
     * @return \Illuminate\Http\Response
     */
    public function edit(Link $link)
    {
        $categories = Category::all();
        return view('links.edit', compact('link', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Link  $link
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Link $link)
    {
        if(auth()->user()->id !== $link->user_id){
            abort(401, "unauthorized");
        }
        
        //validate
        $validated = $this->validate($request, [
            'title' => 'required|min:5|max:150',
            'url' => 'nullable|url|max:255',
            'category_id' => 'required',
            'icon' => 'sometimes|image|mimes:png,jpg,jpeg,gif,webp|max:2048',
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string|max:255',
            'description' => 'required|min:20'
        ]);

        // Icon update and deleting from the storage
        if($request->hasFile('icon')){
            $filename = time() . '.' . $request->icon->getClientOriginalExtension();
            if($link->icon){
                Storage::disk('public')->delete('icon/' . $link->icon);
            }
            $request->icon->storeAs('icon', $filename, 'public');
            
            // $link->icon = $filename;
            $validated['icon'] = $filename;
        }

        // Keep urlslug in sync when URL changes.
        if (!empty($validated['url'])) {
            $rawUrl = (string) $validated['url'];
            $parts = parse_url($rawUrl);
            $host = $parts['host'] ?? '';
            $path = $parts['path'] ?? '';
            $validated['urlslug'] = ltrim($host . $path, '/');
        }

        //update
        // $link->update($request->all());
        $link->update($validated);

        //redirect
        return redirect()->route('link.show', $link->slug)->withMessage('Post description has updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Link  $link
     * @return \Illuminate\Http\Response
     */
    public function destroy(Link $link)
    {
        $link->delete();
        // Star::where('link_id', $link)->delete();
        return redirect()->route('home')->withMessage('Record deleted successfully!');
    }
}
